//////////
//
//	File:		WinFramework.c
//
//	Contains:	Basic support for playing QuickTime and QuickTime VR movies in a Windows application.
//				WinFramework is a simple QuickTime viewer and editor framework. It handles windows,
//				menus, messages, and other low-level things. Put your application-specific code into
//				the file ComApplication.c (or other files).
//
//	Written by:	Tim Monroe
//				Based on MDIPlayer sample code by Brian S. Friedkin (Aug 5, 1996).
//
//	Copyright:	 1997-1999 by Apple Computer, Inc., all rights reserved.
//
//	Change History (most recent first):
//
//	   <45>	 	04/02/99	rtm		added default name for new movies; added DoSaveAsMovieWindow to implement
//									correct "Save As" behavior; added SetWindowTitleFromFSSpec; tweaked
//									GetDisplayName to handle case when file doesn't exist yet
//	   <44>	 	03/18/99	rtm		added SetMoviePlayHints call in DoCreateMovieWindow to support possible
//									dynamic resizing of streamed movies
//	   <43>	 	02/01/99	rtm		added Save As menu command to basic shell
//	   <42>	 	11/02/98	rtm		made changes to compile with CodeWarrior IDE 3.1: WinMain takes HINSTANCE;
//	   								reworked DoCreateMovieWindow to take movie name from the FSSpec when
//									a movie is passed in
//	   <41>	 	10/20/98	rtm		added default progress procedure for all movies (in DoCreateMovieWindow)
//	   <40>	 	10/15/98	rtm		added autostarting for streamed movies
//	   <39>	 	05/18/98	rtm		renamed DoSaveMovie to DoUpdateMovieFile, to mirror MacFramework.c
//	   <38>	 	05/06/98	rtm		added CheckQuickTimeRegistration call to WinMain
//	   <37>	 	03/19/98	rtm		added !IsIconic(theWnd) before call to MCIsPlayerEvent in MovieWndProc
//									to fix drawing problems when arranging icons; added DoApplicationEventLoopAction
//									call to MovieWndProc
//	   <36>	 	03/18/98	rtm		added !IsIconic(theWnd) to test in WM_SIZE message processing in MovieWndProc
//									to fix sizing problems when minimized; added WM_WINDOWPOSCHANGED processing to
//									MovieWndProc to stop movies from playing in minimized windows
//	   <35>	 	03/16/98	rtm		fixed DoOpenCommandLineMovies
//	   <34>	 	03/14/98	rtm		added NativePathNameToFSSpec call to WinMain; changed OpenResFile to FSpOpenResFile
//	   <33>	 	02/28/98	rtm		removed call to DoApplicationEventLoopAction from main message loop
//	   <32>	 	02/13/98	rtm		added GetWindowWidth; reworked SizeWindowToMovie to call it
//	   <31>	 	02/09/98	rtm		added GetWindowReferenceFromPort
//	   <30>	 	02/06/98	rtm		added code to open the application's resource file (see WinMain);
//									removed WM_MDIACTIVATE and WM_MOUSEACTIVATE handling from MovieWndProc
//	   <29>	 	02/02/98	rtm		added SetMenuState
//	   <28>	 	01/06/98	rtm		added WM_ACTIVATE handling to FrameWndProc, and added both WM_MDIACTIVATE
//									and WM_MOUSEACTIVATE handling to MovieWndProc
//	   <27>	 	12/18/97	rtm		added QuitFramework, to mirror MacFramework.c
//	   <26>	 	12/12/97	rtm		added WM_MOUSEMOVE processing to MovieWndProc
//	   <25>	 	12/11/97	rtm		added GetDisplayName; finished reworking DoCreateMovieWindow
//	   <24>	 	12/10/97	rtm		started reworking DoCreateMovieWindow
//	   <23>	 	12/09/97	rtm		added DoOpenCommandLineMovies to handle movies dropped onto application icon
//	   <22>	 	12/08/97	rtm		added DoCautionAlert; finished support for Save; added Select All to Edit menu;
//									added call to SHAddToRecentDocs in DoCreateMovieWindow
//	   <21>	 	12/05/97	rtm		added fDirty flag to window object; begun supporting Save and Save As
//	   <20>	 	12/03/97	rtm		fixed centering of Open File dialog box; see DialogProc
//	   <19>	 	11/06/97	rtm		fixed SizeWindowToMovie (now QT windows hide controller bar correctly)
//	   <18>	 	11/05/97	rtm		removed DoIdleProcessing; fixed bugs in DoCreateMovieWindow, AdjustMenus
//	   <17>	 	11/04/97	rtm		added HandleApplicationMenu call to FrameWndProc
//	   <16>	 	10/30/97	rtm		added GetPortFromWindowReference
//	   <15>	 	10/24/97	rtm		simplified SizeWindowToMovie, following latest MDIPlayer.c; removed
//									GetWindowBorderWidth and GetWindowBorderHeight (called only by SizeWindowToMovie)
//	   <14>	 	10/23/97	rtm		moved InitializeQTVR and TerminateQTVR to ComApplication.c
//	   <13>	 	10/16/97	rtm		removed idle timer; removed MyGetMessage (QT emits idle events, eh?)
//	   <12>	 	10/14/97	rtm		upgraded to latest header files
//	   <11>	 	09/18/97	rtm		added GetFrontWindow and GetNextWindow; added gIdleTimer to emit idles
//	   <10>	 	09/15/97	rtm		added INIT_QTVR and TERM_QTVR to WinMain
//	   <9>	 	09/10/97	rtm		added MyGetMessage to send WM_IDLE messages when no messages are in queue;
//									used EnumChildWindows to process idle messages for all movie windows
//	   <8>	 	09/10/97	rtm		changed DoOpenMovie into DoCreateMovieWindow and SetupMovieWindowWithController
//									(based on functions of same names in VRShell's MacFramework.c)
//	   <7>	 	09/08/97	rtm		conditionalized code for creating grow box
//	   <6>	 	08/22/97	rtm		added code to DialogProc to center dialog boxes (from SimplePlayerSDI);
//									factored out GetFile from DoOpenMovie; rewrote DoIdleProcessing
//	   <5>	 	08/21/97	rtm		changed child windows' icon spec to NULL; added WM_CLOSE case to DialogProc;
//									added hooks for WinApplication.c; added GetAppDataFromFrontWindow et al.
//	   <4>	 	08/20/97	rtm		removed "CALLBACK" from declaration of ApplicationMCActionFilterProc and
//									changed return type to Boolean; seems to fix some QuickTime movie problems;
//									added GetMessagePos and GetMessageTime to event handling
//	   <3>	 	08/18/97	rtm		removed "&& gNumWindowsOpen" test from WM_INITMENU in FrameWndProc;
//									added LOWORD stripping to WM_COMMAND switches (accelerators were being lost)
//	   <2>	 	07/03/97	rtm		made changes to compile with latest QT3.0 headers;
//									ported to CodeWarrior IDE version 2.0
//	   <1>	 	07/02/97	rtm		first file; revised to personal coding style
//	   
//////////

// application header files
#include "WinFramework.h"
#include "ComResource.h"

// Windows header files
#include <shlobj.h>										// for SHAddToRecentDocs

// C runtime header files
#include <stdlib.h>
#include <malloc.h>

// global variables
Rect				gLimitRect = {10, 10, 480, 640};	// max size for any window

BOOL				gShuttingDown = false;				// flag that keeps track of termination state
HANDLE				ghInst;								// the instance of this application
HWND				ghWnd;								// the MDI frame window; this window has the menu bar
HWND				ghWndMDIClient; 					// the MDI client window

char				gChildName[] = "QTShellChild";
char				gMovieType[] = "QuickTime Movie";
Rect				gMCResizeBounds;
BOOL				gShowGrowBox = true;
int					gNumWindowsOpen = 0;
BOOL				gWeAreSizingWindow = false;
BOOL				gWeAreCreatingWindow = false;

short 				gAppResFile = kInvalidFileRefNum;	// file reference number for this application's resource file

char				gAppName[20];						// the name of this application
LPSTR				gCmdLine;							// the command line passed to WinMain

ModalFilterUPP		gModalFilterUPP = NULL;				// UPP to our custom dialog event filter


//////////
//
// WinMain
// The main function for this application.
//
//////////

int CALLBACK WinMain (HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR theCmdLine, int nCmdShow)
{
	HANDLE				myAccel;
	HWND				myWindowFrame;
	MSG					myMsg;
    WNDCLASSEX			myWC;
	char				myFileName[MAX_PATH];
	DWORD				myLength;

	ghInst = hInstance;
	gCmdLine = theCmdLine;
	
	if (hPrevInstance == NULL) {
		LoadString(hInstance, IDS_APPNAME, gAppName, sizeof(gAppName));
		
		// register the frame window class
		myWC.cbSize        = sizeof(WNDCLASSEX);
		myWC.style         = CS_HREDRAW | CS_VREDRAW;
		myWC.lpfnWndProc   = (WNDPROC)FrameWndProc;
		myWC.cbClsExtra    = 0;
		myWC.cbWndExtra    = 0;
		myWC.hInstance     = hInstance;
		myWC.hIcon         = LoadIcon(hInstance, MAKEINTRESOURCE(IDI_APPICON));
		myWC.hCursor       = LoadCursor(NULL, IDC_ARROW);
		myWC.hbrBackground = (HBRUSH)(COLOR_WINDOW + 1);
		myWC.lpszMenuName  = gAppName;
		myWC.lpszClassName = gAppName;
		myWC.hIconSm       = LoadImage(hInstance, MAKEINTRESOURCE(IDI_APPICON), IMAGE_ICON, 16, 16, 0);
									 
		if (!RegisterClassEx(&myWC)) {
			if (!RegisterClass((LPWNDCLASS)&myWC.style))
        		return(0);
		}

		// register the movie child window class
		myWC.cbSize        = sizeof(WNDCLASSEX);
		myWC.style         = 0;
		myWC.lpfnWndProc   = (WNDPROC)MovieWndProc;
		myWC.cbClsExtra    = 0;
		myWC.cbWndExtra    = 0;
		myWC.hInstance     = hInstance;
		myWC.hIcon         = LoadIcon(hInstance, MAKEINTRESOURCE(IDI_CHILDICON));
		// to avoid having QuickTime VR "fight" with the system over the cursor,
		// we set the client area cursor to NULL; this means that for QuickTime
		// movies, we'll need to change the cursor to an arrow manually; see the
		// handling of the WM_MOUSEMOVE message in MovieWndProc
		myWC.hCursor       = NULL;
		myWC.hbrBackground = (HBRUSH)(COLOR_WINDOW + 1);
		myWC.lpszMenuName  = NULL;
		myWC.lpszClassName = gChildName;
		myWC.hIconSm       = LoadImage(hInstance, MAKEINTRESOURCE(IDI_CHILDICON), IMAGE_ICON, 16, 16, 0);
									 
		if (!RegisterClassEx(&myWC)) {
			if (!RegisterClass((LPWNDCLASS)&myWC.style))
        		return(0);
		}
	}

	// load accelerators
	myAccel = LoadAccelerators(hInstance, gAppName);

	// initialize QuickTime Media Layer
	InitializeQTML(0L);
	
	// initialize QuickTime
	EnterMovies();
	
	// if you are distributing QuickTime 3, you might need to call CheckQuickTimeRegistration
	// to satisfy the license agreement; you don't need to do this if you are not bundling
	// QuickTime 3 with your product; check your license agreement for complete details....
	// CheckQuickTimeRegistration(NULL, 0L);
	
	// get the application's resource file, if it exists
	myLength = GetModuleFileName(NULL, myFileName, MAX_PATH);		// NULL means: the current process
	if (myLength != 0) {
		FSSpec			myFSSpec;
		
		NativePathNameToFSSpec(myFileName, &myFSSpec, 0L);

		gAppResFile = FSpOpenResFile(&myFSSpec, fsRdWrPerm);
		if (gAppResFile != kInvalidFileRefNum)
			UseResFile(gAppResFile);
	}

	// do any application-specific initialization that must occur before the frame window is created
	InitApplication(kInitAppPhase_BeforeCreateFrameWindow);
	
	// create the main frame window
	myWindowFrame = CreateWindow(gAppName, gAppName, WS_OVERLAPPEDWINDOW | WS_CLIPCHILDREN,
                               		CW_USEDEFAULT, 
                               		CW_USEDEFAULT,
                             		CW_USEDEFAULT, 
                             		CW_USEDEFAULT,
                             		NULL, 
                             		NULL, 
                             		hInstance, 
                             		NULL);
	ghWnd = myWindowFrame;
	
	// make sure we got a frame window
	if (myWindowFrame == NULL)
		return(0);
		
	// show the window
	ShowWindow(myWindowFrame, nCmdShow);
	UpdateWindow(myWindowFrame);
	
	// do any application-specific initialization that must occur after the frame window is created
	InitApplication(kInitAppPhase_AfterCreateFrameWindow);
	
	// process messages
    while (GetMessage(&myMsg, NULL, 0, 0)) {	
		if (!TranslateMDISysAccel(ghWndMDIClient, &myMsg)) {
        	if (!TranslateAccelerator(myWindowFrame, myAccel, &myMsg)) {
                TranslateMessage(&myMsg);
                DispatchMessage(&myMsg);
			}
		}
    }

	// close the application's resource file, if it was previously opened
	if (gAppResFile != kInvalidFileRefNum)
		CloseResFile(gAppResFile);

	// terminate the QuickTime Media Layer
	ExitMovies();
	TerminateQTML();

	return(myMsg.wParam);			// returns the value from PostQuitMessage
}


//////////
//
// FrameWndProc
// The window procedure for the MDI frame window.
//
//////////

LRESULT CALLBACK FrameWndProc (HWND theWnd, UINT theMessage, UINT wParam, LONG lParam)
{
	HWND       			myChild;

	switch (theMessage) {
	
		case WM_CREATE: {
			CLIENTCREATESTRUCT		myClientStruct = {0};

			myClientStruct.hWindowMenu  = GetSubMenu(GetMenu(theWnd), WINDOWMENU);
			myClientStruct.idFirstChild = IDM_WINDOWCHILD;
			
			// create the MDI client filling the client area
			ghWndMDIClient = CreateWindow("mdiclient",
										 NULL,
										 WS_CHILD | WS_CLIPCHILDREN | WS_VSCROLL | WS_HSCROLL,
										 0, 0, 0, 0,
										 theWnd,
										 (HMENU)0xCAC,
										 ghInst,
										 (LPVOID)&myClientStruct);
			
			// set initial menu state
			AdjustMenus(NULL, GetMenu(theWnd));
			
			if (ghWndMDIClient != NULL)
				ShowWindow(ghWndMDIClient, SW_SHOW);
			
			return(0);
		}

		case WM_ACTIVATE:
			// the MDI frame window is being activated or deactivated;
			// activate or deactivate any active child window by sending this message to DefMDIChildProc 
			myChild = (HWND)SendMessage(ghWndMDIClient, WM_MDIGETACTIVE, 0, 0L);
			if (IsWindow(myChild))
				SendMessage(myChild, WM_ACTIVATE, wParam, lParam);
			break;

		case WM_COMMAND: {

			switch (LOWORD(wParam)) {
				case IDM_FILENEW:
					DoCreateNewMovie();
					break;

				case IDM_FILEOPEN:
					DoCreateMovieWindow(NULL, NULL);
					break;

				case IDM_FILESAVE:
				case IDM_FILESAVEAS:
					// save the active child window
					myChild = (HWND)SendMessage(ghWndMDIClient, WM_MDIGETACTIVE, 0, 0L);
					if (IsWindow(myChild))
						SendMessage(myChild, WM_COMMAND, wParam, lParam);
					return(0);

				case IDM_FILECLOSE:
					// close the active child window
					myChild = (HWND)SendMessage(ghWndMDIClient, WM_MDIGETACTIVE, 0, 0L);
					if (IsWindow(myChild))
						SendMessage(myChild, WM_CLOSE, 0L, 0L);
					return(0);

				case IDM_EXIT:
					// set our global flag to indicate we're shutting down
					gShuttingDown = true;
					
					// do application-specific processing that must occur before movie windows are closed
					StopApplication(kStopAppPhase_BeforeDestroyWindows);
					
					// close all open movie windows;
					// note that the user can cancel the shutting down
					SendMessage(theWnd, WM_COMMAND, (WPARAM)IDM_WINDOWCLOSEALL, 0L);
					
					// close the frame window, if we're still shutting down
					if (gShuttingDown)
						SendMessage(theWnd, WM_CLOSE, 0, 0L);
						
					return(0);

				case IDM_WINDOWTILE:
					SendMessage(ghWndMDIClient, WM_MDITILE, 0, 0L);
					return(0);

				case IDM_WINDOWCASCADE:
					SendMessage(ghWndMDIClient, WM_MDICASCADE, 0, 0L);
					return(0);

				case IDM_WINDOWICONS:
					SendMessage(ghWndMDIClient, WM_MDIICONARRANGE, 0, 0L);
					return(0);

				case IDM_WINDOWCLOSEALL: {
					WindowReference		myWindow, myNextWindow;
			
					// walk the window list and destroy any open windows
					myWindow = GetFrontMovieWindow();
					while (myWindow != NULL) {
						myNextWindow = GetNextMovieWindow(myWindow);
						SendMessage(myWindow, WM_CLOSE, 0L, 0L);
						myWindow = myNextWindow;
					}
					
					return(0);
				}

				case IDM_ABOUT:
					ShowAboutBox();
					return(0);

				default:				
					// pass this message to the active child window...
					myChild = (HWND)SendMessage(ghWndMDIClient, WM_MDIGETACTIVE, 0, 0L);
					if (IsWindow(myChild))
						SendMessage(myChild, WM_COMMAND, wParam, lParam);

					// ...then do any application-specific menu handling, if no movie windows are open...
					if (myChild == NULL)
						HandleApplicationMenu((UInt16)LOWORD(wParam));
					
					// ...and then pass it to DefFrameProc
					break;
			}
			break;
		}

		case WM_OPENDROPPEDFILES:
			// open any movie files that were dropped onto the application icon
			DoOpenCommandLineMovies(gCmdLine);
			return(0);

		case WM_INITMENU:
			if (GetMenu(theWnd) == (HMENU)wParam)
				return(AdjustMenus((HWND)SendMessage(ghWndMDIClient, WM_MDIGETACTIVE, 0, 0L), (HMENU)wParam));
			return(1);

		case WM_CLOSE:
			// if we're not already in the process of shutting down,
			// simulate the selection of the Quit menu command
			if (!gShuttingDown) {
				SendMessage(ghWnd, WM_COMMAND, IDM_EXIT, 0L);
				return(0);
			}
			break;
			
		case WM_DESTROY:
			// do any application-specific shutdown
			StopApplication(kStopAppPhase_AfterDestroyWindows);
			PostQuitMessage(0);
			break;
	}
	
	return(DefFrameProc(theWnd, ghWndMDIClient, theMessage, wParam, lParam));
}


//////////
//
// MovieWndProc
// The window procedure for a movie window.
//
//////////

LRESULT CALLBACK MovieWndProc (HWND theWnd, UINT theMessage, UINT wParam, LONG lParam)
{
	WPARAM				myWidth, myHeight;
	MovieController		myMC = NULL;
	Movie				myMovie = NULL;
	WindowObject		myWindowObject = NULL;
	MSG					myMsg = {0};

	// get the window object, movie, and movie controller for this window
	myWindowObject = GetWindowObjectFromWindow(theWnd);
	if (myWindowObject != NULL) {
		myMC = (**myWindowObject).fController;
		myMovie = (**myWindowObject).fMovie;
	}

	// give the movie controller this message first
	if (!gShuttingDown && myMC) {
		EventRecord		myMacEvent;
		LONG			myPoints = GetMessagePos();

		myMsg.hwnd = theWnd;
		myMsg.message = theMessage;
		myMsg.wParam = wParam;
		myMsg.lParam = lParam;
		myMsg.time = GetMessageTime();
		myMsg.pt.x = LOWORD(myPoints);
		myMsg.pt.y = HIWORD(myPoints);

		// translate a Windows event to a Mac event
		WinEventToMacEvent(&myMsg, &myMacEvent);

		DoApplicationEventLoopAction(&myMacEvent);
		
		// pass the Mac event to the movie controller, but only if the movie window isn't minimized
		if (!IsIconic(theWnd))
			MCIsPlayerEvent(myMC, (EventRecord *)&myMacEvent);
	}

	switch (theMessage) {
		case WM_CREATE:
			// create some private storage
			myWindowObject = (WindowObject)NewHandleClear(sizeof(WindowObjectRecord));
			if (myWindowObject != NULL) {
				(**myWindowObject).fWindow = theWnd;
				(**myWindowObject).fController = NULL;
				(**myWindowObject).fObjectType = kMovieControllerObject;
				(**myWindowObject).fInstance = NULL;
				(**myWindowObject).fAppData = NULL;
			}

			SetWindowLong(theWnd, GWL_USERDATA, (LPARAM)myWindowObject);

			// associate a GrafPort with this window and set the port
			CreatePortAssociation(theWnd, NULL, 0L);
			MacSetPort(GetPortFromWindowReference(theWnd));
			break;

		case WM_WINDOWPOSCHANGING:
			// don't show the window until we have created a movie and
			// can therefore properly size the window to contain the movie
			if (gWeAreCreatingWindow) {
				WINDOWPOS	*lpWindowPos = (WINDOWPOS*)lParam;
				
				lpWindowPos->flags &= ~SWP_SHOWWINDOW;
			}
			break;

		case WM_WINDOWPOSCHANGED:
			// if a movie window has become minimized, stop the movie
			if (IsIconic(theWnd))
				StopMovie(myMovie);
			break;

		case WM_SIZE:
			// resize the movie and controller to fit the window
			myWidth = LOWORD(lParam);
			myHeight = HIWORD(lParam);
			
			// we do NOT want to resize the movie controller if the window is minimized,
			// if there is no movie controller, or if we are in the middle of resizing the window
			if (!gWeAreSizingWindow && (myMC != NULL) && !IsIconic(theWnd)) {
				Rect		myRect;
				
				myRect.top = 0;
				myRect.left = 0;
				myRect.right = myWidth;
				myRect.bottom = myHeight;
				
				MCSetControllerBoundsRect(myMC, &myRect);
			}
			break;

		case WM_MOUSEMOVE:
			// for QuickTime movies (but NOT for QuickTime VR movies), set the cursor to the arrow cursor
			if (myWindowObject != NULL)
				if ((**myWindowObject).fInstance == NULL)
					SetCursor(LoadCursor(NULL, IDC_ARROW));
			break;

		case WM_PUMPMOVIE:
			// we receive this message only to idle the movie
			break;

		case WM_CHAR:
			// do any application-specific key press handling
			HandleApplicationKeyPress((char)wParam);
			break;

		case WM_COMMAND: {

			switch (LOWORD(wParam)) {
							
				case IDM_FILESAVE:
					DoUpdateMovieFile(theWnd);
					break;

				case IDM_FILESAVEAS:
					DoSaveAsMovieWindow(theWnd);
					break;

				case IDM_EDITUNDO:
					DoUndo(theWnd);
					break;

				case IDM_EDITCUT:
					DoCut(theWnd);
					break;

				case IDM_EDITCOPY:
					DoCopy(theWnd);
					break;

				case IDM_EDITPASTE:
					DoPaste(theWnd);
					break;

				case IDM_EDITCLEAR:
					DoClear(theWnd);
					break;
					
				case IDM_EDITSELECTALL:
					if (myMC != NULL)
						QTUtils_SelectAllMovie(myMC);
					break;
					
				default:
					// do any application-specific menu handling
					HandleApplicationMenu((UInt16)LOWORD(wParam));
					break;
			}
			
			break;
		}	// case WM_COMMAND

		case WM_GETMINMAXINFO:
			CalcWindowMinMaxInfo(theWnd, (LPMINMAXINFO)lParam);
			break;

		case WM_CLOSE:
			// prepare to close the window, making sure that any changed data is saved or explicitly discarded;
			// we can still cancel the window closing here
			if (myWindowObject != NULL) {
			
				// if the window's data is "dirty", give the user a chance to save it
				if ((**myWindowObject).fDirty) {
					int			myItem;
					char		myText[256];
		
					// get the title of the window
					GetWindowText(theWnd, myText, sizeof(myText));
		
					// display the "Save changes" dialog box
					myItem = DoCautionAlert(theWnd, IDS_SAVEDIALOG, MB_ICONEXCLAMATION, MB_YESNOCANCEL, gAppName, myText);
					switch (myItem) {
						case kSaveChanges:
							DoUpdateMovieFile(theWnd);	// save the data in the window
							break;
							
						case kDontSaveChanges:
							break;						// discard any unsaved changes (that is, don't do anything)
							
						case kCancelClose:
							gShuttingDown = false;		// do not close the window and do not quit the application
							return(0);
						
						default:
							return(0);					// unexpected item selected; just return
					}
				}
			} // if (myWindowObject != NULL)
			
			// if we got to this point, it's okay to close and destroy the window
			SendMessage(ghWndMDIClient, WM_MDIDESTROY, (WPARAM)theWnd, 0L);
			break;

		case WM_DESTROY:
			// when we get this message,
			// the window has been removed from the screen and must be destroyed; no turning back!
			if (myWindowObject != NULL) {
								
				// dispose movie and controller
				if (myMC != NULL) {
					MCSetActionFilterWithRefCon(myMC, NULL, 0);
					DisposeMovieController(myMC);
				}
				
				if (myMovie != NULL)
					DisposeMovie(myMovie);
					
				// do any application-specific window clean-up, then toss the window object
				RemoveApplicationWindowObject(myWindowObject);
				DisposeHandle((Handle)myWindowObject);
			}
		
			SetWindowLong(theWnd, GWL_USERDATA, 0);

			// destroy the port association
			DestroyPortAssociation((CGrafPtr)GetHWNDPort(theWnd));
			
			// decrement the count of open movie windows, if there are any
			if (gNumWindowsOpen > 0)
				gNumWindowsOpen--;

			break;
	}

	return(DefMDIChildProc(theWnd, theMessage, wParam, lParam));
}


//////////
//
// GetFile
// Display the file-opening dialog box and return the name of the selected file (if any).
//
//////////

BOOL GetFile (char *theFileName)
{
    OPENFILENAME    myOFN = {0};

	// fill in the OPENFILENAME structure
	memset(&myOFN, 0, sizeof(OPENFILENAME));	// first zero out the structure
	*theFileName = '\0';
    myOFN.lStructSize = sizeof(OPENFILENAME);
    myOFN.hwndOwner = NULL;
    myOFN.lpstrFile = (LPSTR)theFileName;
    myOFN.nMaxFile  = 255;
	myOFN.lpstrFilter  = "QuickTime Movies (*.mov) \0 *.mov\0All Files (*.*) \0 *.*\0";
    myOFN.nFilterIndex = 1;
    myOFN.lpstrInitialDir = NULL;
	if (USEEXPLORERSTYLE)
		myOFN.Flags = OFN_PATHMUSTEXIST | OFN_FILEMUSTEXIST | OFN_ENABLEHOOK | OFN_EXPLORER;
	else
		myOFN.Flags = OFN_PATHMUSTEXIST | OFN_FILEMUSTEXIST | OFN_ENABLEHOOK;
    myOFN.lCustData = kOpenDialogCustomData;
	myOFN.lpfnHook = DialogProc;

	// present the file-opening dialog
    if (GetOpenFileName(&myOFN))
        return(true);
	else
        return(false);
}


//////////
//
// AdjustMenus 
// Adjust the application's menus.
//
// The theWnd parameter is a handle to the active MDI *child* window, if any.
//
//////////

int AdjustMenus (HWND theWnd, HMENU theMenu)
{
	WindowObject		myWindowObject = NULL; 
	MovieController		myMC = NULL;

	if (theWnd != NULL)
		myWindowObject = GetWindowObjectFromWindow(theWnd);
	
	if (myWindowObject != NULL)
		myMC = (**myWindowObject).fController;

	// configure the edit menu
	if (myMC != NULL) {
		long			myFlags;

		MCGetControllerInfo(myMC, &myFlags);

		EnableMenuItem(theMenu, IDM_EDITUNDO, myFlags & mcInfoUndoAvailable ? MF_ENABLED : MF_GRAYED);
		EnableMenuItem(theMenu, IDM_EDITCUT, myFlags & mcInfoCutAvailable ? MF_ENABLED : MF_GRAYED);
		EnableMenuItem(theMenu, IDM_EDITCOPY, myFlags & mcInfoCopyAvailable ? MF_ENABLED : MF_GRAYED);
		EnableMenuItem(theMenu, IDM_EDITPASTE, myFlags & mcInfoPasteAvailable ? MF_ENABLED : MF_GRAYED);
		EnableMenuItem(theMenu, IDM_EDITCLEAR, myFlags & mcInfoClearAvailable ? MF_ENABLED : MF_GRAYED);
		EnableMenuItem(theMenu, IDM_EDITSELECTALL, myFlags & mcInfoEditingEnabled ? MF_ENABLED : MF_GRAYED);
	} else {
		EnableMenuItem(theMenu, IDM_EDITUNDO, MF_GRAYED);
		EnableMenuItem(theMenu, IDM_EDITCUT, MF_GRAYED);
		EnableMenuItem(theMenu, IDM_EDITCOPY, MF_GRAYED);
		EnableMenuItem(theMenu, IDM_EDITPASTE,MF_GRAYED);
		EnableMenuItem(theMenu, IDM_EDITCLEAR, MF_GRAYED);
		EnableMenuItem(theMenu, IDM_EDITSELECTALL, MF_GRAYED);
	}
	
	// enable the Close item if there are any movie windows opened, else disable
	EnableMenuItem(theMenu, IDM_FILECLOSE, (gNumWindowsOpen) ? MF_ENABLED : MF_GRAYED);

	// enable the Save As item if there are any movie windows opened, else disable
	EnableMenuItem(theMenu, IDM_FILESAVEAS, (gNumWindowsOpen) ? MF_ENABLED : MF_GRAYED);

	// enable the Save item if the movie window is dirty, else disable
	if (myWindowObject != NULL)
		EnableMenuItem(theMenu, IDM_FILESAVE, ((**myWindowObject).fDirty) ? MF_ENABLED : MF_GRAYED);
	else
		EnableMenuItem(theMenu, IDM_FILESAVE, MF_GRAYED);

	// adjust any application-specific menus
	AdjustApplicationMenus(theWnd, theMenu);

	return(0);
}


//////////
//
// QuitFramework
// Do any framework-specific shut-down.
//
//////////

void QuitFramework (void)
{
	// send a message to the MDI frame window telling it to close down and exit the application
	SendMessage(ghWnd, WM_COMMAND, IDM_EXIT, 0L);
}


//////////
//
// DoCreateNewMovie
// Create a new movie in a window; returns true if successful.
//
// NOTE: There are several user interface issues that are blissfully ignored by this routine,
// principally the preferred names and the on-screen locations of the new windows. 
//
//////////

Boolean DoCreateNewMovie (void)
{
	Movie				myMovie = NULL;
	FSSpec				myFSSpec;
	StringPtr 			myName = QTUtils_ConvertCToPascalString(kNewMovieName);
	
	myMovie = NewMovie(newMovieActive);
	if (myMovie == NULL)
		return(false);
	
	// create a default FSSpec
	FSMakeFSSpec(0, 0L, myName, &myFSSpec);
	
	free(myName);
	
	return(DoCreateMovieWindow(myMovie, &myFSSpec));
}


//////////
//
// SetupMovieWindowWithController
// Configure the movie controller.
//
//////////

MovieController SetupMovieWindowWithController (Movie theMovie, WindowReference theWindow, Boolean theMoveWindow)
{
#pragma unused(theMoveWindow)

	MovieController			myMC = NULL;
	Rect					myRect;
	WindowObject			myWindowObject = NULL;
	
	if ((theMovie == NULL) || (theWindow == NULL))
		return(NULL);
		
	myWindowObject = GetWindowObjectFromWindow(theWindow);		// get our window specific data

	// resize the movie bounding rect and offset to 0,0
	GetMovieBox(theMovie, &myRect);
	MacOffsetRect(&myRect, -myRect.left, -myRect.top);
	SetMovieBox(theMovie, &myRect);
	AlignWindow(GetHWNDPort(theWindow), false, &myRect, NULL);

	// create the movie controller
	myMC = NewMovieController(theMovie, &myRect, mcTopLeftMovie);
	if (myMC == NULL)
		return(NULL);
		
	// enable the default movie controller editing
	MCEnableEditing(myMC, true);
		
	// suppress movie badge
	MCDoAction(myMC, mcActionSetUseBadge, (void *)false);

	// set the initial looping state of the movie
	QTUtils_SetLoopingStateFromFile(theMovie, myMC);
	
	// install an action filter that does any application-specific movie controller action processing
	MCSetActionFilterWithRefCon(myMC, NewMCActionFilterWithRefConProc(ApplicationMCActionFilterProc), (long)myWindowObject);

	// add grow box for the movie controller
	if (gShowGrowBox) {
		RECT				myRect;

		GetWindowRect(GetDesktopWindow(), &myRect);
		
		OffsetRect(&myRect, -myRect.left, -myRect.top);
		gMCResizeBounds.top = (short)myRect.top;
		gMCResizeBounds.left = (short)myRect.left;
		gMCResizeBounds.right = (short)myRect.right;
		gMCResizeBounds.bottom = (short)myRect.bottom;

		MCDoAction(myMC, mcActionSetGrowBoxBounds, &gMCResizeBounds);
	}
	
	// add any application-specific controller functionality
	AddControllerFunctionality(myMC);
		
	return(myMC);
}


//////////
//
// DoOpenCommandLineMovies
// Parse the command line when the application first starts up and
// open as movie documents any files specified on the command line.
//
// Based on the routine ParseCmdLinePriv in GraphicImporter.c.
//
//////////

void DoOpenCommandLineMovies (LPSTR theCmdLine)
{
#pragma unused(theCmdLine)
	LPSTR				myCmdLine;
	FSSpec				myFSSpec;
	SHFILEINFO			myFileInfo;
	
	// get the command line for the current process
	myCmdLine = GetCommandLine();

	// parse the command line
	if (*myCmdLine) {
		LPSTR			myTempLine;
		
		// the string preceding any white space is the name of the module (that is, the application)
		myTempLine = strchr(myCmdLine, ' ');
		if (myTempLine) {
			myCmdLine = myTempLine;  				// skip the name of the application
			while (*myCmdLine == ' ')
				myCmdLine++;            			// skip spaces to end of string or to first command

			while (*myCmdLine != '\0') {
				char 	myFileName[MAX_PATH];
				char 	myTempName[MAX_PATH];
				char 	myBuffName[MAX_PATH];
				int 	myIndex;
				
				// read thru the remaining string to find file names
				for (myIndex = 0; *myCmdLine != '\0'; myIndex++, myCmdLine++) {
					// if we encounter a space character, it might be a filename delimiter or a space in the filename;
					// we'll try to open the filename we have so far to see whether it's a valid filename; if not, the
					// space must be part of the filename we're constructing
					if (*myCmdLine == ' ') {
						HANDLE				myFindFile;
						WIN32_FIND_DATA		myFile;
					
						myTempName[myIndex] = '\0';
						strcpy(myBuffName, myTempName);
						
						myFindFile = FindFirstFile(myBuffName, &myFile);
						if (myFindFile != INVALID_HANDLE_VALUE) {
							// we found a file having the specified name; close our file search and
							// break out of our character-gobbling loop (since we've got a valid filename)
							FindClose(myFindFile);
							break;
						}
					}
				
					// if we made it here, *myCmdLine is part of the filename (possibly a space)
					myFileName[myIndex] = myTempName[myIndex] = *myCmdLine;
				}
				
				if (*myCmdLine != '\0')
					myCmdLine++;
				
				// add a terminating NULL character
				myFileName[myIndex] = '\0';

				// make sure the filename picks out a QuickTime movie
				SHGetFileInfo(myFileName, (DWORD)0, &myFileInfo, sizeof(myFileInfo), SHGFI_TYPENAME);
				if (strcmp(myFileInfo.szTypeName, gMovieType) != 0)
					continue;
				
				// make an FSSpec record
				NativePathNameToFSSpec(myFileName, &myFSSpec, 0L);

				// open the file in a movie window
				DoCreateMovieWindow(NULL, &myFSSpec);
			}

		} else
			myCmdLine += strlen(myCmdLine);   		// point to NULL
	}
}


//////////
//
// DoCreateMovieWindow
// Open a movie in a new movie window; returns true if successful.
//
// This function is called from several places in our framework. The following combinations are possible:
//	* theMovie == NULL, theFSSpec == NULL: no movie, no file; elicit a movie file from user and open it
//	* theMovie != NULL, theFSSpec == NULL: new movie, no file (yet)
//	* theMovie == NULL, theFSSpec != NULL: no movie, but we have an FSSpec; so just open the specified movie file
//	* theMovie != NULL, theFSSpec != NULL: new movie, theFSSpec contains (at least) the movie name
//
//////////

BOOL DoCreateMovieWindow (Movie theMovie, FSSpec *theFSSpec)
{
	WindowObject		myWindowObject = NULL;
	MovieController		myMC = NULL;
	WindowReference		myWindow = NULL;
	FSSpec				myFSSpec;
	Movie				myMovie = NULL;
	short				myRefNum = kInvalidFileRefNum;
	short				myResID = 0;
    DWORD				myVersion;
    char           		myPathName[MAX_PATH];				// the full pathname of the file
    char				myFileName[MAX_PATH];				// the movie file name
	GrafPtr				mySavedPort;
	OSErr				myErr = noErr;

	// get the current port; we may need to restore it if we cannot successfully create a new window
	GetPort(&mySavedPort);

	// if we got neither a movie nor an FSSpec passed in, prompt the user for a movie file
	if ((theMovie == NULL) && (theFSSpec == NULL)) {
	
		// display file-opening dialog box
		if (!GetFile(myPathName))
			goto bail;

		// add this document to the Documents list
		SHAddToRecentDocs(SHARD_PATH, myPathName);

		// get the movie file name from the full pathname
		GetDisplayName(myPathName, myFileName);
		
		// make an FSSpec record
		FSMakeFSSpec(0, 0L, c2pstr(myPathName), &myFSSpec);
	}
	
	// if we got an FSSpec passed in, copy it into myFSSpec
	if (theFSSpec != NULL) {
	
		// make an FSSpec record
		FSMakeFSSpec(theFSSpec->vRefNum, theFSSpec->parID, theFSSpec->name, &myFSSpec);	
	}
	
	// if we got no movie passed in, read one from the specified file
	if (theMovie == NULL) {

		// ideally, we'd like read and write permission, but we'll settle for read-only permission
		myErr = OpenMovieFile(&myFSSpec, &myRefNum, fsRdWrPerm);
		if (myErr != noErr)
			myErr = OpenMovieFile(&myFSSpec, &myRefNum, fsRdPerm);

		// if we couldn't open the file with even just read-only permission, bail....
		if (myErr != noErr)
			goto bail;

		// now fetch the first movie from the file
		myResID = 0;
		myErr = NewMovieFromFile(&myMovie, myRefNum, &myResID, NULL, newMovieActive, NULL);
		if (myErr != noErr)
			goto bail;
	
		CloseMovieFile(myRefNum);
		
	} else {
		myMovie = theMovie;
	}

	//////////
	//
	// at this point, myMovie is an open movie, but myFSSpec may or may not be a valid FSSpec
	//
	//////////
	
	// set the default progress procedure for the movie
	SetMovieProgressProc(myMovie, (MovieProgressUPP)-1, 0);
		
	// create a new window to display the movie in
	gWeAreCreatingWindow = true;
	
	// create the child movie window
	myVersion = GetVersion();
	if ((myVersion < 0x80000000) || (LOBYTE(LOWORD(myVersion)) < 4)) {
		// this is Windows NT or Win32s, so use the WM_MDICREATE message
		MDICREATESTRUCT 	mcs;

		mcs.szClass = gChildName;
		mcs.szTitle = myFileName;
		mcs.hOwner  = ghInst;
		mcs.x       = CW_USEDEFAULT;
		mcs.y       = CW_USEDEFAULT;
		mcs.cx      = CW_USEDEFAULT;
		mcs.cy      = CW_USEDEFAULT;
		mcs.style   = 0;
		mcs.lParam  = 0;

		myWindow = (HWND)SendMessage(ghWndMDIClient,
									WM_MDICREATE,
									0,
									(LPARAM)(LPMDICREATESTRUCT)&mcs);
	} else {
		// this method will only work with Windows 95, not Windows NT or Win32s
		myWindow = CreateWindowEx(WS_EX_MDICHILD,
								   gChildName,
								   myFileName,
								   0,
								   CW_USEDEFAULT,
								   CW_USEDEFAULT,
								   CW_USEDEFAULT,
								   CW_USEDEFAULT,
								   ghWndMDIClient, 
								   NULL,
								   ghInst,
								   0);
	}
	
	gWeAreCreatingWindow = false;
	if (myWindow == NULL)
		goto bail;

	MacSetPort(GetPortFromWindowReference(myWindow));
	myWindowObject = GetWindowObjectFromWindow(myWindow);
	if (myWindowObject == NULL)
		goto bail;

	// set the window title
	SetWindowTitleFromFSSpec(myWindow, &myFSSpec, true);

	// make sure the movie uses the window GWorld in all situations
	SetMovieGWorld(myMovie, (CGrafPtr)GetPortFromWindowReference(myWindow), GetGWorldDevice((CGrafPtr)GetPortFromWindowReference(myWindow)));

	// create and configure the movie controller
	myMC = SetupMovieWindowWithController(myMovie, myWindow, true);

	// store movie info in the window record
	(**myWindowObject).fMovie = myMovie;
	(**myWindowObject).fController = myMC;
	(**myWindowObject).fFileResID = myResID;
	(**myWindowObject).fFileRefNum = myRefNum;
	(**myWindowObject).fCanResizeWindow = true;
	(**myWindowObject).fDirty = false;
	(**myWindowObject).fInstance = NULL;
	(**myWindowObject).fAppData = NULL;
	(**myWindowObject).fFileFSSpec = myFSSpec;
	
	// do any application-specific window object initialization
	InitApplicationWindowObject(myWindowObject);
	
	// size the window to fit the movie and controller
	SizeWindowToMovie(myWindowObject);
		
	// set the movie's play hints to allow dynamic resizing
	SetMoviePlayHints(myMovie, hintsAllowDynamicResize, hintsAllowDynamicResize);

	// show the window
	ShowWindow(myWindow, SW_SHOW);
	UpdateWindow(myWindow);

	// one more movie window has been opened
	gNumWindowsOpen++;

	// if the movie is a streamed movie, then start it playing immediately
	if (QTUtils_IsStreamedMovie(myMovie))
		MCDoAction(myMC, mcActionPrerollAndPlay, (void *)GetMoviePreferredRate(myMovie));
		
	return(true);

bail:
	if (myWindow != NULL)
		SendMessage(ghWndMDIClient, WM_MDIDESTROY, (WPARAM)myWindow, 0L);
		
	if (myMC != NULL)
		DisposeMovieController(myMC);
		
	if (myMovie != NULL)
		DisposeMovie(myMovie);
		
	if (myRefNum != 0)
		CloseMovieFile(myRefNum);
		
	MacSetPort(mySavedPort);	// restore the port that was active when this function was called

	return(false);
}


//////////
//
// DoSaveAsMovieWindow
// Save the movie in the specified window under a new name.
//
//////////

void DoSaveAsMovieWindow (WindowReference theWindow)
{
	WindowObject		myWindowObject = NULL;
	Movie 				myMovie = NULL;
	StandardFileReply	mySFReply;
	StringPtr 			myPrompt = QTUtils_ConvertCToPascalString(kSavePrompt);
	StringPtr 			myFileName = QTUtils_ConvertCToPascalString(kSaveMovieFileName);
	
	// get the window object associated with the specified window
	myWindowObject = GetWindowObjectFromWindow(theWindow);
	if (myWindowObject == NULL)
		goto bail;
		
	myMovie = (**myWindowObject).fMovie;
	if (myMovie == NULL)
		goto bail;
		
	StandardPutFile(myPrompt, myFileName, &mySFReply); 
	if (mySFReply.sfGood) {
		FlattenMovieData(myMovie, flattenAddMovieToDataFork, &mySFReply.sfFile, FOUR_CHAR_CODE('TVOD'), smSystemScript, createMovieFileDeleteCurFile);
		
		// keep track of the new file specification
		(**myWindowObject).fFileFSSpec = mySFReply.sfFile;
		
		// set the window title
		SetWindowTitleFromFSSpec(theWindow, &mySFReply.sfFile, true);
	}

bail:
	free(myPrompt);
	free(myFileName);
}


//////////
//
// DoUpdateMovieFile
// Update the file (if any) attached to the movie.
//
//////////

Boolean DoUpdateMovieFile (WindowReference theWindow)
{
	WindowObject		myWindowObject = NULL;
	Movie 				myMovie = NULL;
	OSErr				myErr = noErr;
	
	// get the window object associated with the specified window
	myWindowObject = GetWindowObjectFromWindow(theWindow);
	if (myWindowObject == NULL)
		return(false);
		
	myMovie = (**myWindowObject).fMovie;
	if (myMovie == NULL)
		return(false);
		
	if ((**myWindowObject).fFileRefNum == kInvalidFileRefNum) {		// brand new movie, so no file attached to it
		if (QTUtils_SaveMovie(myMovie) != noErr)
			return(false);	
	} else {														// we have an existing file; just update the movie resource
		// open the movie resource file, update the resource, and then close it again
		myErr = OpenMovieFile(&(**myWindowObject).fFileFSSpec, &(**myWindowObject).fFileRefNum, fsRdWrPerm);
		if (myErr != noErr)
			return(false);
		
		myErr = UpdateMovieResource(myMovie, (**myWindowObject).fFileRefNum, (**myWindowObject).fFileResID, NULL);
		CloseMovieFile((**myWindowObject).fFileRefNum);
	}
	
	(**myWindowObject).fDirty = false;

	return(myErr == noErr);
}


//////////
//
// DoCut
// Cut a movie segment.
//
//////////

static void DoCut (HWND theWnd)
{
	MovieController		myMC = NULL;
	WindowObject		myWindowObject = NULL;
	
	// get the window object associated with the specified window
	myWindowObject = GetWindowObjectFromWindow(theWnd);
	if (myWindowObject == NULL)
		return;

	myMC = (**myWindowObject).fController;
	if (myMC != NULL) {
		Movie			myMovie;
		
		myMovie = MCCut(myMC);				// cut the segment
		if (myMovie != NULL) {
			PutMovieOnScrap(myMovie, 0L);	// place the segment into the scrap
			DisposeMovie(myMovie);
		}
		
		(**myWindowObject).fDirty = true;
	}
}


//////////
//
// DoCopy
// Copy a movie segment.
//
//////////

static void DoCopy (HWND theWnd)
{
	MovieController		myMC = NULL;
	WindowObject		myWindowObject = NULL;
	
	// get the window object associated with the specified window
	myWindowObject = GetWindowObjectFromWindow(theWnd);
	if (myWindowObject == NULL)
		return;

	myMC = (**myWindowObject).fController;
	if (myMC != NULL) {
		Movie			myMovie;
		
		myMovie = MCCopy(myMC);				// copy the segment
		if (myMovie != NULL) {
			PutMovieOnScrap(myMovie, 0L);	// place the segment into the scrap
			DisposeMovie(myMovie);
		}
	}
}


//////////
//
// DoPaste
// Paste a movie segment.
//
//////////

static void DoPaste (HWND theWnd)
{
	MovieController		myMC = NULL;
	WindowObject		myWindowObject = NULL;
	
	// get the window object associated with the specified window
	myWindowObject = GetWindowObjectFromWindow(theWnd);
	if (myWindowObject == NULL)
		return;

	myMC = (**myWindowObject).fController;
	if (myMC != NULL) {
		MCPaste(myMC, NULL);
		(**myWindowObject).fDirty = true;
	}
}


//////////
//
// DoClear
// Clear a movie segment.
//
//////////

static void DoClear (HWND theWnd)
{
	MovieController		myMC = NULL;
	WindowObject		myWindowObject = NULL;
	
	// get the window object associated with the specified window
	myWindowObject = GetWindowObjectFromWindow(theWnd);
	if (myWindowObject == NULL)
		return;

	myMC = (**myWindowObject).fController;
	if (myMC != NULL) {
		MCClear(myMC);
		(**myWindowObject).fDirty = true;
	}
}


//////////
//
// DoUndo
// Undo an editing operation.
//
//////////

static void DoUndo (HWND theWnd)
{
	MovieController		myMC = NULL;
	WindowObject		myWindowObject = NULL;
	
	// get the window object associated with the specified window
	myWindowObject = GetWindowObjectFromWindow(theWnd);
	if (myWindowObject == NULL)
		return;

	myMC = (**myWindowObject).fController;
	if (myMC != NULL) {
		MCUndo(myMC);
		(**myWindowObject).fDirty = true;
	}
}


//////////
//
// GetDisplayName
// Given a full pathname, return the part that trails the rightmost path separator,
// in long file name format (not in 8.3 format).
//
//////////

static void GetDisplayName (char *thePathName, char *theDispName)
{
	SHFILEINFO			myFileInfo;
	DWORD				myResult;
	
	myResult = SHGetFileInfo(thePathName, (DWORD)0, &myFileInfo, sizeof(myFileInfo), SHGFI_DISPLAYNAME);
	if (myResult != 0) {
		// SHGetFileInfo successful
		strcpy(theDispName, myFileInfo.szDisplayName);
	} else {
		// SHGetFileInfo not successful, so find the basename ourselves
		short	myLength = 0;
		short	myIndex;

		// get the length of the pathname
		myLength = strlen(thePathName);
		
		// find the position of the rightmost path separator in thePathName
		if (strchr(thePathName, kWinFilePathSeparator) != NULL) {
	
			myIndex = myLength - 1;
			while (thePathName[myIndex] != kWinFilePathSeparator)
				myIndex--;
				
			// calculate the length of the basename
			myLength = myLength - myIndex - 1;
	
		} else {
			// there is no rightmost path separator in thePathName;
			// set myIndex so that myIndex + 1 == 0, for the call to BlockMove below
			myIndex = -1;
		}
		
		// copy into theDispName the substring of thePathName from myIndex + 1 to the end
		BlockMove(&thePathName[myIndex + 1], theDispName, myLength);
		theDispName[myLength] = '\0';
	}
}


//////////
//
// SizeWindowToMovie
// Set the window size to exactly fit the movie and controller (if visible).
//
//////////

void SizeWindowToMovie (WindowObject theWindowObject)
{
	Rect				myMovieBounds;
	MovieController		myMC = NULL;
	Movie				myMovie = NULL;

	gWeAreSizingWindow = true;

	if (theWindowObject == NULL)
		goto bail;
	
	myMC = (**theWindowObject).fController;
	myMovie = (**theWindowObject).fMovie;

	if (MCGetVisible(myMC))
		MCGetControllerBoundsRect(myMC, &myMovieBounds);
	else
		GetMovieBox(myMovie, &myMovieBounds);
	
	// make sure that the movie has a non-zero width;
	// a zero height is okay (for example, with a music movie with no controller bar)
	if (myMovieBounds.right - myMovieBounds.left == 0) {
		myMovieBounds.left = 0;
		myMovieBounds.right = GetWindowWidth((**theWindowObject).fWindow);
	}
	
	SizeWindow(GetPortFromWindowReference((**theWindowObject).fWindow),
											myMovieBounds.right - myMovieBounds.left,
											myMovieBounds.bottom - myMovieBounds.top,
											true);

bail:												
	gWeAreSizingWindow = false;
}


//////////
//
// ShowAboutBox 
// Display and manage the About dialog box.
//
//////////

static void ShowAboutBox (void)
{
	DialogBox(ghInst, MAKEINTRESOURCE(IDD_ABOUT), ghWnd, (DLGPROC)DialogProc);
}


//////////
//
// DoCautionAlert 
// Display and manage a caution alert.
//
// Based on ShowUserMessage by Stephen Chernicoff, in his WiniEdit application
// (as described in the book "From Mac to Windows" on CodeWarrior reference CD).
//
//////////

int DoCautionAlert (HWND theWnd, UINT theID, UINT theIconStyle, UINT theButtonStyle, LPSTR theTitle, LPSTR theArgument)
{
	char			myTemplate[kAlertMessageMaxLength];
	char			myText[kAlertMessageMaxLength];
	UINT			myStyle;
	int				myItem;
	
	// beep, to get the user's attention (just like CautionAlert on MacOS)
	DoBeep();
	
	// load the message text template from a resource
	LoadString(ghInst, theID, myTemplate, sizeof(myTemplate));
	
	// insert argument into the message text template, to get the message text
	wsprintf(myText, myTemplate, theArgument);
	
	// set the dialog box style
	myStyle = theIconStyle | theButtonStyle | MB_APPLMODAL | MB_SETFOREGROUND;

	// display the dialog box
	myItem = MessageBox(theWnd, myText, theTitle, myStyle);
	
	return(myItem);
}


//////////
//
// DialogProc 
// Dialog callback procedure.
//
//////////

static UINT APIENTRY DialogProc (HWND theDialog, UINT theMessage, WPARAM wParam, LPARAM lParam)
{
	BOOL	isHandled = false;

	switch (theMessage) {
	
		case WM_INITDIALOG: {
			Point			myPoint;
			long			myWidth;
			long			myHeight;
			RECT			myRect;
			RECT			myDeskRect;
			HWND			myWindow;
			OPENFILENAME	*myOFNPtr = (OPENFILENAME *)lParam;
			
			myWindow = theDialog;
				
			// check whether theDialog is the Open File common dialog box
			
			// we need to do this because, for the Open File dialog box, theDialog isn't
			// the actual visible dialog box, but an invisible child of the visible dialog box;
			// for WM_INITDIALOG, lParam is the address of the structure passed to GetOpenFileName,
			// so we can just look for the value we previously put into the lCustData field
			// to make sure that we've got the correct dialog.
			if (myOFNPtr != NULL)
				if (myOFNPtr->lCustData == kOpenDialogCustomData)
					myWindow = GetParent(theDialog);
				
			// center the dialog window on the screen
			GetWindowRect(myWindow, &myRect);
			myWidth = myRect.right - myRect.left;
			myHeight = myRect.bottom - myRect.top;
			GetWindowRect(GetDesktopWindow(), &myDeskRect);
			myPoint.h = (short)((myDeskRect.right + myDeskRect.left)/2 - myWidth/2);
			myPoint.v = (short)((myDeskRect.top + myDeskRect.bottom)/3 - myHeight/3);
			SetWindowPos(myWindow, 0, myPoint.h, myPoint.v, 0, 0, SWP_NOZORDER | SWP_NOSIZE);
			
			// let Windows set the input focus
			isHandled = true;
			break;
		}
		
		case WM_CLOSE:
			EndDialog(theDialog, IDOK);
			isHandled = true;
			break;
		
		case WM_COMMAND:
			switch (LOWORD(wParam)) {
				case IDOK:
					EndDialog(theDialog, IDOK);
					isHandled = true;
					break;
				default:
					isHandled = false;
					break;
			}
			break;
		
		default:
			isHandled = false;
			break;
	}

	return(isHandled);
}


//////////
//
// CalcWindowMinMaxInfo 
// Get minimum and maximum possible size of this window.
//
//////////

static void CalcWindowMinMaxInfo (HWND theWnd, LPMINMAXINFO lpMinMax)
{
	WindowObject		myWindowObject = NULL;
	MovieController		myMC = NULL;
	Movie				myMovie = NULL;

	myWindowObject = GetWindowObjectFromWindow(theWnd);
	if (myWindowObject != NULL) {
		myMC = (**myWindowObject).fController;
		myMovie = (**myWindowObject).fMovie;
	}

	if (myMC && myMovie) {
		Rect			myMovieBox;
		short			myControllerHeight = 0;					// assume there's no controller bar
		
		if (MCGetVisible(myMC))
			myControllerHeight = QTUtils_GetControllerBarHeight(myMC);

		lpMinMax->ptMinTrackSize.x = gMCResizeBounds.left + (2 * GetSystemMetrics(SM_CXFRAME));
		
		GetMovieBox(myMovie, &myMovieBox);
		if (myMovieBox.bottom - myMovieBox.top != 0)
			lpMinMax->ptMinTrackSize.y = 
				gMCResizeBounds.top +							// growbounds height +
				(2 * GetSystemMetrics(SM_CXFRAME)) +			// frame thickness +
				GetSystemMetrics(SM_CYCAPTION) +				// caption height +
				-1 +											// fudge factor +
				myControllerHeight;								// movie controller bar height
		else
			lpMinMax->ptMaxSize.y =
			lpMinMax->ptMaxTrackSize.y =
			lpMinMax->ptMinTrackSize.y = 
				0 +												// height of audio only movie +
				(2 * GetSystemMetrics(SM_CXFRAME)) +			// frame thickness +
				GetSystemMetrics(SM_CYCAPTION) +				// caption height +
				-1 +											// fudge factor +
				myControllerHeight;								// movie controller bar height
	}
}


//////////
//
// SetWindowTitleFromFSSpec
// Set the title of the specified window, using the name contained in the specified FSSpec.
//
//////////

static void SetWindowTitleFromFSSpec (HWND theWnd, FSSpec *theFSSpec, Boolean theAddToRecentDocs)
{
	char	*myTempName;
	char	myWindName[MAX_PATH];

	// get the full pathname contained in the FSSpec (which is a Str255)
	myTempName = QTUtils_ConvertPascalToCString(theFSSpec->name);

	// get the movie file name from the full pathname
	GetDisplayName(myTempName, myWindName);

	// set the window title
	SetWindowText(theWnd, myWindName);
	
	if (theAddToRecentDocs) {
		// add this document to the Documents list
		SHAddToRecentDocs(SHARD_PATH, myTempName);
	}
	
	free(myTempName);
}


///////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Window walking utilities.
//
// Use these two functions to iterate through all open movie windows belonging to the application.
//
///////////////////////////////////////////////////////////////////////////////////////////////////////////

//////////
//
// GetFrontMovieWindow
// Return a reference to the frontmost movie window.
//
//////////

WindowReference GetFrontMovieWindow (void)
{
	return(GetWindow(ghWndMDIClient, GW_CHILD));
}


//////////
//
// GetNextMovieWindow
// Return a reference to the next movie window.
//
//////////

WindowReference GetNextMovieWindow (WindowReference theWindow)
{
	return(GetWindow(theWindow, GW_HWNDNEXT));
}


///////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Data retrieval utilities.
//
// Use the following functions to retrieve the window object, the application-specific data, or the movie
// controller that is associated with a window.
//
///////////////////////////////////////////////////////////////////////////////////////////////////////////

//////////
//
// GetWindowObjectFromFrontWindow
// Get the window object associated with the frontmost child window.
//
//////////

WindowObject GetWindowObjectFromFrontWindow (void)
{
	HWND               myChild;

	myChild = (HWND)SendMessage(ghWndMDIClient, WM_MDIGETACTIVE, 0, 0L);
	return(GetWindowObjectFromWindow(myChild));
}


//////////
//
// GetWindowObjectFromWindow
// Get the window object associated with the specified window.
//
//////////

WindowObject GetWindowObjectFromWindow (WindowReference theWnd)
{
	WindowObject		myWindowObject;

	if (theWnd == NULL)
		return(NULL);
				
	myWindowObject = (WindowObject)GetWindowLong(theWnd, GWL_USERDATA);

	// make sure this is a window object
	if (!IsWindowObjectOurs(myWindowObject))
		return(NULL);
		
	return(myWindowObject);
}


//////////
//
// GetMCFromFrontWindow
// Get the movie controller (if any) associated with the frontmost child window.
//
//////////

MovieController GetMCFromFrontWindow (void)
{
	MovieController 	myMC = NULL;
	WindowObject		myWindowObject = NULL;
		
	myWindowObject = GetWindowObjectFromFrontWindow();
	if (myWindowObject != NULL)
		myMC = (**myWindowObject).fController;
		
	return(myMC);
}


//////////
//
// GetMCFromWindow
// Get the movie controller (if any) associated with the specified window.
//
//////////

MovieController GetMCFromWindow (WindowReference theWindow)
{
	MovieController 	myMC = NULL;
	WindowObject		myWindowObject = NULL;
		
	myWindowObject = GetWindowObjectFromWindow(theWindow);
	if (myWindowObject != NULL)
		myMC = (**myWindowObject).fController;
		
	return(myMC);
}


//////////
//
// GetQTVRInstanceFromFrontWindow
// Get the QTVRInstance (if any) associated with the frontmost child window.
//
//////////

QTVRInstance GetQTVRInstanceFromFrontWindow (void)
{
	QTVRInstance	 	myInstance = NULL;
	WindowObject		myWindowObject = NULL;

	myWindowObject = GetWindowObjectFromFrontWindow();
	if (myWindowObject != NULL)
		myInstance = (**myWindowObject).fInstance;
		
	return(myInstance);
}


//////////
//
// GetAppDataFromFrontWindow
// Get the application-specific data associated with the frontmost child window.
//
//////////

Handle GetAppDataFromFrontWindow (void)
{
	HWND               myChild;

	myChild = (HWND)SendMessage(ghWndMDIClient, WM_MDIGETACTIVE, 0, 0L);
	return(GetAppDataFromWindow(myChild));
}


//////////
//
// GetAppDataFromWindow
// Get the application-specific data associated with the specified window.
//
//////////

Handle GetAppDataFromWindow (WindowReference theWnd)
{
	WindowObject		myWindowObject = NULL;
	
	myWindowObject = GetWindowObjectFromWindow(theWnd);
	if (myWindowObject == NULL)
		return(NULL);
	else
		return(GetAppDataFromWindowObject(myWindowObject));
}


//////////
//
// GetAppDataFromWindowObject
// Get the application-specific data associated with the specified window object.
//
//////////

Handle GetAppDataFromWindowObject (WindowObject theWindowObject)
{
	Handle				myAppData = NULL;
			
	// make sure this is a window object belonging to our application
	if (!IsWindowObjectOurs(theWindowObject))
		return(myAppData);
	
	// get the app data handle from the window object
	myAppData = (**theWindowObject).fAppData;
	
	return(myAppData);
}


//////////
//
// IsWindowObjectOurs
// Does the specified window object belong to our application?
//
//////////

Boolean IsWindowObjectOurs (WindowObject theWindowObject)
{
	OSType		myType = 0;

	if ((theWindowObject == NULL) || (*theWindowObject == NULL))
		return(false);
		
	myType = (**theWindowObject).fObjectType;
	if (myType == kMovieControllerObject)
		return(true);
	else
		return(false);
}


///////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Miscellaneous utilities.
//
// Use the following functions to play beeps, manipulate menus, and do other miscellaneous things. These
// functions are intended for use in cross-platform code.
//
///////////////////////////////////////////////////////////////////////////////////////////////////////////

//////////
//
// DoBeep
// Beep.
//
//////////

void DoBeep (void)
{
	MessageBeep(MB_OK);
}


//////////
//
// SetMenuState
// Set the enabled/disabled state of a menu.
//
//////////

void SetMenuState (MenuReference theMenu, UInt16 theMenuRank, short theState)
{
	SetMenuItemState(theMenu, theMenuRank, theState | MF_BYPOSITION);
}


//////////
//
// SetMenuItemState
// Set the enabled/disabled state of a menu item.
//
//////////

void SetMenuItemState (MenuReference theMenu, UInt16 theMenuItem, short theState)
{
	EnableMenuItem(theMenu, (UINT)theMenuItem, (UINT)theState);
}


//////////
//
// SetMenuItemLabel
// Set the label (that is, the text) of a menu item.
//
//////////

void SetMenuItemLabel (MenuReference theMenu, UInt16 theMenuItem, char *theText)
{
	ModifyMenu(theMenu, (UINT)theMenuItem, MF_BYCOMMAND | MF_STRING, (UINT)theMenuItem, (LPSTR)theText);
}


//////////
//
// SetMenuItemCheck
// Set the check mark state state of a menu item.
//
//////////

void SetMenuItemCheck (MenuReference theMenu, UInt16 theMenuItem, Boolean theState)
{
	CheckMenuItem(theMenu, (UINT)theMenuItem, theState ? MF_CHECKED : MF_UNCHECKED);
}


//////////
//
// GetPortFromWindowReference 
// Return the graphics port associated with a window reference.
//
//////////

GrafPtr GetPortFromWindowReference (WindowReference theWindow)
{
	GrafPtr		myPort = NULL;
	
	if (theWindow != NULL)
		myPort = (GrafPtr)GetHWNDPort(theWindow);
	
	return(myPort);
}


//////////
//
// GetWindowReferenceFromPort
// Return the window reference associated with a graphics port.
//
//////////

WindowReference GetWindowReferenceFromPort (GrafPtr thePort)
{
	WindowReference		myWindow = NULL;
	
	if (thePort != NULL)
		myWindow = (WindowReference)GetPortHWND(thePort);
	
	return(myWindow);
}


//////////
//
// GetWindowWidth
// Return the width of the specified window.
//
//////////

short GetWindowWidth (WindowReference theWindow)
{
	short		myWidth = 0;
	RECT		myRect;
	
	if (theWindow != NULL)
		if (GetWindowRect(theWindow, &myRect))
			myWidth = myRect.right - myRect.left;

	return(myWidth);
}
